import { useState } from 'react'
import { useNavigate } from 'react-router-dom'
import { Check, User, Video, AlertCircle, Heart, Briefcase, Calendar, Clock, Shield, FileText, Award, Users } from 'lucide-react'
import './BookSession.css'

const BookSession = () => {
  const navigate = useNavigate()
  const [selectedFormat, setSelectedFormat] = useState(null)
  const [selectedTopic, setSelectedTopic] = useState(null)
  const [selectedDate, setSelectedDate] = useState(null)

  const formats = [
    { id: 'in-person', name: 'Лично', icon: User },
    { id: 'video', name: 'Видео', icon: Video }
  ]

  const topics = [
    { id: 'stress', name: 'Стресс', icon: AlertCircle, color: '#ef4444' },
    { id: 'relationships', name: 'Отношения', icon: Heart, color: '#ec4899' },
    { id: 'career', name: 'Карьера', icon: Briefcase, color: '#f59e0b' }
  ]

  const generateDates = () => {
    const dates = []
    for (let i = 0; i < 14; i++) {
      const date = new Date()
      date.setDate(date.getDate() + i)
      dates.push(date.toISOString().split('T')[0])
    }
    return dates
  }

  const handleSubmit = (e) => {
    e.preventDefault()
    
    if (!selectedFormat || !selectedTopic || !selectedDate) {
      alert('Пожалуйста, заполните все поля')
      return
    }

    const sessions = JSON.parse(localStorage.getItem('psychologistSessions') || '{"past": []}')
    const formatName = formats.find(f => f.id === selectedFormat)?.name || selectedFormat
    const topicName = topics.find(t => t.id === selectedTopic)?.name || selectedTopic
    
    const newSession = {
      id: Date.now(),
      date: selectedDate,
      type: formatName,
      topic: topicName,
      rating: null
    }

    sessions.past.push(newSession)
    localStorage.setItem('psychologistSessions', JSON.stringify(sessions))
    
    alert('Сеанс успешно записан!')
    navigate('/')
  }

  return (
    <div className="book-session">
      <div className="booking-hero">
        <div className="booking-hero-content">
          <div className="booking-hero-top">
            <span className="booking-hero-badge">Запись на консультацию</span>
            <h1 className="booking-hero-title">Выберите параметры сеанса</h1>
            <p className="booking-hero-subtitle">Определите формат встречи, выберите тему, которая вас интересует, и назначьте удобную дату</p>
          </div>
          <div className="booking-hero-features">
            <div className="booking-feature">
              <div className="booking-feature-dot"></div>
              <span>Безопасность данных</span>
            </div>
            <div className="booking-feature">
              <div className="booking-feature-dot"></div>
              <span>Профессионализм</span>
            </div>
            <div className="booking-feature">
              <div className="booking-feature-dot"></div>
              <span>Удобство</span>
            </div>
          </div>
        </div>
        <div className="booking-hero-visual">
          <div className="visual-element visual-element-1"></div>
          <div className="visual-element visual-element-2"></div>
          <div className="visual-element visual-element-3"></div>
        </div>
      </div>

      <form className="booking-form" onSubmit={handleSubmit}>
        <div className="form-section">
          <h2 className="section-title">
            <User className="section-icon" size={24} />
            Выберите формат консультации
          </h2>
          <div className="formats-grid">
            {formats.map(format => {
              const IconComponent = format.icon
              return (
                <button
                  key={format.id}
                  type="button"
                  className={`format-card ${selectedFormat === format.id ? 'selected' : ''}`}
                  onClick={() => setSelectedFormat(format.id)}
                >
                  <div className="format-icon">
                    <IconComponent className="format-icon-img" size={40} />
                  </div>
                  <span className="format-name">{format.name}</span>
                </button>
              )
            })}
          </div>
        </div>

        <div className="form-section">
          <h2 className="section-title">
            <FileText className="section-icon" size={24} />
            Выберите тему консультации
          </h2>
          <div className="topics-grid">
            {topics.map(topic => {
              const IconComponent = topic.icon
              return (
                <button
                  key={topic.id}
                  type="button"
                  className={`topic-card ${selectedTopic === topic.id ? 'selected' : ''}`}
                  onClick={() => setSelectedTopic(topic.id)}
                  style={{ '--topic-color': topic.color }}
                >
                  <div className="topic-icon">
                    <IconComponent className="topic-icon-img" size={32} />
                  </div>
                  <span className="topic-name">{topic.name}</span>
                </button>
              )
            })}
          </div>
        </div>

        <div className="form-section">
          <h2 className="section-title">
            <Calendar className="section-icon" size={24} />
            Выберите дату сеанса
          </h2>
          <div className="dates-scroll">
            {generateDates().map(date => {
              const dateObj = new Date(date)
              const isToday = date === new Date().toISOString().split('T')[0]
              return (
                <button
                  key={date}
                  type="button"
                  className={`date-button ${isToday ? 'today' : ''} ${selectedDate === date ? 'selected' : ''}`}
                  onClick={() => setSelectedDate(date)}
                >
                  <div className="date-day">{dateObj.getDate()}</div>
                  <div className="date-weekday">{dateObj.toLocaleDateString('ru-RU', { weekday: 'short' })}</div>
                </button>
              )
            })}
          </div>
        </div>

        <button type="submit" className="submit-button" disabled={!selectedFormat || !selectedTopic || !selectedDate}>
          Записаться на консультацию
        </button>
      </form>

      <div className="benefits-showcase">
        <div className="benefits-header">
          <h2 className="benefits-title">Наши преимущества</h2>
          <p className="benefits-subtitle">Что делает нас лучшим выбором</p>
        </div>
        <div className="benefits-grid">
          <div className="benefit-item">
            <div className="benefit-icon-wrapper">
              <Shield className="benefit-icon" size={32} />
            </div>
            <h3>Конфиденциальность</h3>
            <p>Строгая защита персональных данных</p>
          </div>
          <div className="benefit-item">
            <div className="benefit-icon-wrapper">
              <Award className="benefit-icon" size={32} />
            </div>
            <h3>Опыт специалистов</h3>
            <p>Сертифицированные психологи</p>
          </div>
          <div className="benefit-item">
            <div className="benefit-icon-wrapper">
              <Users className="benefit-icon" size={32} />
            </div>
            <h3>Гибкость</h3>
            <p>Онлайн и офлайн форматы</p>
          </div>
        </div>
      </div>

      <div className="info-section">
        <div className="info-section-header">
          <h2 className="info-title">Полезная информация</h2>
        </div>
        <div className="info-grid">
          <div className="info-card">
            <div className="info-card-icon-wrapper">
              <Clock className="info-card-icon" size={32} />
            </div>
            <h3>Длительность сеанса</h3>
            <p>Стандартная консультация длится 50-60 минут. Возможны расширенные сеансы по договоренности.</p>
          </div>
          <div className="info-card">
            <div className="info-card-icon-wrapper">
              <FileText className="info-card-icon" size={32} />
            </div>
            <h3>Подготовка к сеансу</h3>
            <p>Подготовьте список вопросов и тем, которые хотите обсудить. Для онлайн-сеансов проверьте технику.</p>
          </div>
          <div className="info-card">
            <div className="info-card-icon-wrapper">
              <Shield className="info-card-icon" size={32} />
            </div>
            <h3>Конфиденциальность</h3>
            <p>Все сеансы строго конфиденциальны. Мы гарантируем полную анонимность и защиту ваших данных.</p>
          </div>
        </div>
      </div>
    </div>
  )
}

export default BookSession

